﻿/**
 * Controlador dos tiles, responsável por inicializar todos os pais que existirem na tela e fazer a conexão entre a view e o model dos tiles.
 * 
 * OBS: os tiles são os bloquinhos de filtro. Comumente encontrados na área de trabalho, relatórios e no módulo de analytics.
 */
var tileSystemController = {
    init: function () {
        const fathers = document.querySelectorAll('li.father-tile[data-tileid]');
        if (fathers.length > 0) {
            tileSystemModal.init(Array.prototype.map.call(fathers, father => {
                if (father.getAttribute('data-tileid')) {
                    return father.getAttribute('data-tileid');
                }
            }));
        } else {
            tileSystemModal.init();
        }
        tileSystemView.init();
    },
    handleMutation: function () {
        tileSystemView.init();
    },
    getFathers: function () {
        return tileSystemModal.getFathers();
    }
}

/**
 * Estrutura responsável por armazenar e controlar as informações sobre os tiles.
 */
var tileSystemModal = {
    fathers: [],
    init: function (initFathers = []) {
        tileSystemModal.fathers = initFathers.filter(father => father);
    },
    getFathers: function () {
        return tileSystemModal.fathers;
    }
}

/**
 * Estrutura responsável por interagir com a view dos tiles.
 */
var tileSystemView = {
    init: function () {
        const fathers = tileSystemController.getFathers();
        fathers.forEach(fatherid => {
            const father = document.querySelector(`li.father-tile[data-tileid=${fatherid}]`);
            if (father) {
                tileSystemView.bindEvents(father, fatherid);
                tileSystemView.verifyInitOpenTiles(father, fatherid);
            };
        })
    },
    verifyInitOpenTiles: function (father, id) {
        const children = document.querySelectorAll(`li.child-tile[data-father=${id}]`);
        children.forEach(child => {
            if (child.classList.contains('selected')) {
                tileSystemView.openTiles(father, id);
            }
        })
    },
    bindEvents: function (father, id) {
        father.addEventListener('click', (event) => tileSystemView.handleClick(father, id));
    },
    handleClick: function (father, id) {
        document.querySelectorAll(`li.child-tile`).forEach(child => child.classList.add('animated'));
        if (father.classList.contains('open')) {
            tileSystemView.closeTiles(father, id);
        } else {
            tileSystemView.openTiles(father, id);
        }
    },
    openTiles: function (father, id) {
        father.classList.add('open');
        const children = document.querySelectorAll(`li.child-tile[data-father=${id}]`);
        children.forEach(child => {
            child.classList.remove('close');
        })
    },
    closeTiles: function (father, id) {
        father.classList.remove('open');
        const children = document.querySelectorAll(`li.child-tile[data-father=${id}]`);
        children.forEach(child => {
            child.classList.add('close');
        })
    }
}



/**
 * Inicializa a lib.
 */
document.addEventListener("DOMContentLoaded", function (event) {
    /**
     * Construção do observer e inicialização da lib de tiles.
     */
    tileSystemController.init();

    var targets = document.querySelectorAll('.update-panel');

    // cria uma nova instância de observador
    var observer = new MutationObserver(function (mutations) {
        mutations.forEach(function (mutation) {
            if (mutation.type == 'childList') {
                tileSystemController.handleMutation();
            }
        });
    });

    // configuração do observador:
    var config = { attributes: true, childList: true, characterData: true };

    // passar o nó alvo, bem como as opções de observação
    targets.forEach(target => {
        observer.observe(target, config);
    })
});