﻿
/**
 * Controlador da lib de controle de tabelas responsivas, o objetivo dessa lib é identificar tabelas responsivas e adicionar um scroll lateral virtua (quando necessario).
 * 
 * Essa lib foi criada para resolver o problema que clientes estavam tendo quando a quantidade de dados era muito grande e o usuário não conseguia ver o scroll lateral ao final da tabela.
 * 
 * OBS: Essa lib só funciona no ztemplate2022
 */
const tablesController = {
    init: function () {
        const main = document.querySelector('.new-template main, .wf-admin main');
        if (main) {
            main.addEventListener("scroll", this.handleScroll);

            const resizeObserver = new ResizeObserver(this.handleResize);

            resizeObserver.observe(main); 

            this.initializeTables();
        }
    },
    handleScroll: debounce(function () {
        tablesController.initializeTables();
    }, 50),
    handleResize: function () {
        const tableContainers = document.querySelectorAll('.table-responsive.scrollTreated, .RadGrid.scrollTreated');

        tableContainers.forEach(tableContainer => {
            virtualScrollView.handleResize(tableContainer);
        })
    },
    initializeTables: function () {
        // identifica tabelas responsivas na tela (tabelas que são englabadas por divs .table-responsive ou .RadGrid)
        const tableContainers = document.querySelectorAll('.table-responsive:not(.scrollTreated), .RadGrid:not(.scrollTreated)');

        tableContainers.forEach(tableContainer => {
            responsiveTableView.render(tableContainer);
        })
    }
}

const responsiveTableView = {
    render: function (tableContainer) {
        /** trata a tabela para que o scroll e o header funcionem corretamente.
         * O tratamento consiste em pegar o scrollHeight do container e setar o height do container, para podermos user o position stick no header da tabela.
        */
        if (!tableContainer.classList.contains('scrollTreated')) {

            // trata a tabela
            tableContainer.classList.add('scrollTreated');
            tableContainer.style.height = `${tableContainer.offsetHeight}px`;

            // adiciona evento no scroll do main para tratar o reposicionamento do header
            document.querySelector('main').addEventListener("scroll", () => { this.handleHeader(tableContainer) });

            // da inicio ao tratamento do scroll lateral
            virtualScrollView.render(tableContainer);
        }
    },
    handleHeader: function (tableContainer) {
        const mainRect = document.querySelector("main").getBoundingClientRect();
        /**
        * Identificar se a tabela está visível, se não estiver já segue o jogo.
        *  usar getBoundingClientRect() para verificar o posicionamento da tabela. scrollTop não funcionará porque o scroll é numa div interna
        * 
        * Uma vez que a tabela esteja visível, identificar se o header estaria escondido, se sim, reposicionar o header relativo ao container e
        * adicionar uma classe ao container que adiciona um padding top na tabela do mesmo tamanho do header, para evitar glitches
        * 
        * Em seguida, verificar que o bottom da tabela não está visível, se não estiver, adicionar a scroll bar virtual ao final da página, posicionado no mesmo X da tabela.
        * 
        */
        const { y, height } = tableContainer.getBoundingClientRect();
        const tHead = tableContainer.querySelector('table thead, table.wfreport tbody tr:first-of-type');
        if (tHead) {
            if ((y + 16) < mainRect.y && (y + height) > mainRect.y) { // esse 16 é devido a um padding de 16 que existe nos containers das tabelas.
                tHead.style.top = `${Math.abs(mainRect.y - y) - 17}px`; // 17 equivale ao padding e a borda da tabela 16 + 1
            } else {
                tHead.style.top = 0;
            }
        }

    }
}

const virtualScrollView = {
    render: function (tableContainer) {
        const table = tableContainer.querySelector('table');
        console.log(table);
        console.log(tableContainer);
        if (table) {
             // verifica se o scroll lateral precisa existir
            if (table.clientWidth > tableContainer.clientWidth) {

                // cria os elementos do scroll e seta os estilos
                const scrollContainer = document.createElement('div'); // div que será usada para posicionar o scroll embaixo.
                const relativeDiv = document.createElement('div'); // div que será usada apenas para reposicionar o tracker com position absolute.
                const tracker = document.createElement('div'); // div que será usada para fazer o scroll.
                scrollContainer.classList.add('scrollContainer');
                relativeDiv.classList.add('relativeDiv');
                tracker.classList.add('tracker');
                scrollContainer.style.width = `${tableContainer.clientWidth}px`;
                tracker.style.width = `${(tableContainer.clientWidth * 100) / table.clientWidth}%`;

                // cria eventos para relatizar o scroll
                let mousePosition;
                let offset = 0;
                let isDown = false;

                tracker.addEventListener('mousedown',  (e) => {
                    isDown = true;
                    offset = tracker.offsetLeft - e.clientX;
                    tracker.classList.add('dragging')
                }, true);

                document.addEventListener('mouseup',  () => {
                    isDown = false;
                    tracker.classList.remove('dragging')
                }, true);

                document.addEventListener('mousemove', (event) => {
                    event.preventDefault();
                    if (isDown) {
                        mousePosition = {
                            x: event.clientX,
                            y: event.clientY
                        };
                        let newPosition = mousePosition.x + offset;
                        if (newPosition < 0) {
                            tableContainer.scrollLeft = 0;
                        } else if (newPosition + tracker.clientWidth > scrollContainer.clientWidth) {
                            tableContainer.scrollLeft = table.offsetWidth;
                        } else {
                            tableContainer.scrollLeft = (table.offsetWidth * newPosition) / tableContainer.clientWidth;
                        }
                    }
                }, true);

                tableContainer.addEventListener('scroll', () => {
                    tracker.style.left = tableContainer.scrollLeft > 0 ? `${(tableContainer.scrollLeft * scrollContainer.clientWidth / table.offsetWidth)}px` : 0;
                })

                relativeDiv.appendChild(tracker);
                scrollContainer.appendChild(relativeDiv);
                tableContainer.appendChild(scrollContainer);

                document.querySelector('main').addEventListener("scroll", () => { this.handleScrollVisibility(tableContainer) });

                this.handleScrollVisibility(tableContainer);
            }
        }
    },
    handleScrollVisibility: function (tableContainer) {
        const mainRect = document.querySelector("main").getBoundingClientRect();
        const { y, bottom } = tableContainer.getBoundingClientRect();

        const scrollContainer = tableContainer.querySelector('.scrollContainer');

        if (scrollContainer) {
            if (((y + 16) < (mainRect.y + (mainRect.height - 100))) && (bottom - 40 > mainRect.bottom)) { // o 16 é referente ao padding do container e o 40 é para o scroll sumir antes do real aparecer
                scrollContainer.style.display = 'block';
            } else {
                scrollContainer.style.display = 'none';
            }
        }
    },
    handleResize: function (tableContainer) {
        const table = tableContainer.querySelector('table');
        if (table) {
            const scrollContainer = tableContainer.querySelector('.scrollContainer');
            if (scrollContainer) {
                if (table.clientWidth > tableContainer.clientWidth) {
                    scrollContainer.classList.remove('disabled');
                    const tracker = scrollContainer.querySelector('.tracker')

                    scrollContainer.style.width = `${tableContainer.clientWidth}px`;
                    tracker.style.width = `${(tableContainer.clientWidth * 100) / table.clientWidth}%`;
                } else {
                    scrollContainer.classList.add('disabled');
                }
            }

        }
    }
}


/**
 * Inicializa a lib.
 */
document.addEventListener("DOMContentLoaded", function (event) {
    tablesController.init()
});